<?php
// This file is part of Moodle - http://moodle.org/
//
// Moodle is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// Moodle is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Moodle.  If not, see <http://www.gnu.org/licenses/>.
/**
 * Internal library of functions for choice module.
 *
 * All the choice specific functions, needed to implement the module
 * logic, should go here. Never include this file from your lib.php!
 *
 * @package   mod_signactivityserver
 * @copyright 2016 Stephen Bourget
 * @license   http://www.gnu.org/copyleft/gpl.html GNU GPL v3 or later
 */
defined('MOODLE_INTERNAL') || die();

require('config.php');

/**
 * This creates new calendar events given as timeopen and timeclose by $choice.
 *
 * @param stdClass $choice
 * @return void
 */
function signactivityserver_set_events($choice) {
    global $DB, $CFG;

    require_once($CFG->dirroot.'/calendar/lib.php');

    // Get CMID if not sent as part of $choice.
    if (!isset($choice->coursemodule)) {
        //TODO: check this
        $cm = get_coursemodule_from_instance('signactivityserver', $choice->id, $choice->course);
        $choice->coursemodule = $cm->id;
    }

    // Choice start calendar events.
    $event = new stdClass();
    $event->eventtype = SIG_EVENT_TYPE_OPEN;
    // The SIG_EVENT_TYPE_OPEN event should only be an action event if no close time is specified.
    $event->type = empty($choice->timeclose) ? CALENDAR_EVENT_TYPE_ACTION : CALENDAR_EVENT_TYPE_STANDARD;
    if ($event->id = $DB->get_field('event', 'id',
            array('modulename' => 'signactivityserver', 'instance' => $choice->id, 'eventtype' => $event->eventtype))) {
        if ((!empty($choice->timeopen)) && ($choice->timeopen > 0)) {
            // Calendar event exists so update it.
            $event->name         = get_string('calendarstart', 'choice', $choice->name);
            $event->description  = format_module_intro('signactivityserver', $choice, $choice->coursemodule);
            $event->timestart    = $choice->timeopen;
            $event->timesort     = $choice->timeopen;
            $event->visible      = instance_is_visible('signactivityserver', $choice);
            $event->timeduration = 0;
            $calendarevent = calendar_event::load($event->id);
            $calendarevent->update($event, false);
        } else {
            // Calendar event is on longer needed.
            $calendarevent = calendar_event::load($event->id);
            $calendarevent->delete();
        }
    } else {
        // Event doesn't exist so create one.
        if ((!empty($choice->timeopen)) && ($choice->timeopen > 0)) {
            $event->name         = get_string('calendarstart', 'choice', $choice->name);
            $event->description  = format_module_intro('signactivityserver', $choice, $choice->coursemodule);
            $event->courseid     = $choice->course;
            $event->groupid      = 0;
            $event->userid       = 0;
            $event->modulename   = 'signactivityserver';
            $event->instance     = $choice->id;
            $event->timestart    = $choice->timeopen;
            $event->timesort     = $choice->timeopen;
            $event->visible      = instance_is_visible('signactivityserver', $choice);
            $event->timeduration = 0;
            calendar_event::create($event, false);
        }
    }

    // Choice end calendar events.
    $event = new stdClass();
    $event->type = CALENDAR_EVENT_TYPE_ACTION;
    $event->eventtype = SIG_EVENT_TYPE_CLOSE;
    if ($event->id = $DB->get_field('event', 'id',
            array('modulename' => 'signactivityserver', 'instance' => $choice->id, 'eventtype' => $event->eventtype))) {
        if ((!empty($choice->timeclose)) && ($choice->timeclose > 0)) {
            // Calendar event exists so update it.
            $event->name         = get_string('calendarend', 'choice', $choice->name);
            $event->description  = format_module_intro('signactivityserver', $choice, $choice->coursemodule);
            $event->timestart    = $choice->timeclose;
            $event->timesort     = $choice->timeclose;
            $event->visible      = instance_is_visible('signactivityserver', $choice);
            $event->timeduration = 0;
            $calendarevent = calendar_event::load($event->id);
            $calendarevent->update($event, false);
        } else {
            // Calendar event is on longer needed.
            $calendarevent = calendar_event::load($event->id);
            $calendarevent->delete();
        }
    } else {
        // Event doesn't exist so create one.
        if ((!empty($choice->timeclose)) && ($choice->timeclose > 0)) {
            $event->name         = get_string('calendarend', 'choice', $choice->name);
            $event->description  = format_module_intro('signactivityserver', $choice, $choice->coursemodule);
            $event->courseid     = $choice->course;
            $event->groupid      = 0;
            $event->userid       = 0;
            $event->modulename   = 'signactivityserver';
            $event->instance     = $choice->id;
            $event->timestart    = $choice->timeclose;
            $event->timesort     = $choice->timeclose;
            $event->visible      = instance_is_visible('signactivityserver', $choice);
            $event->timeduration = 0;
            calendar_event::create($event, false);
        }
    }
}

function sig_store_signed_file($name, $content, $userid, $contextid, $filearea) {
    $file = sig_read_signed_file($name, $userid, $contextid, $filearea);
    if($file) {
        $file->delete();
    }

    $fileinfo = array(
        'component' => 'mod_signactivityserver',
        'filearea' => $filearea,
        'itemid' => $userid,
        'mimetype' => 'application/pdf',
        'contextid' => $contextid,
        'filename' => $name,
        'filepath' => '/',
        'userid' => $userid
    );

    $fs = get_file_storage();
    $file = $fs->create_file_from_string($fileinfo, $content);
    return $file;
}

function sig_read_signed_file($name, $userid, $contextid, $filearea) {
    global $DB;
     
    $fileinfo = array(
        'component' => 'mod_signactivityserver',
        'filearea' => $filearea,
        'itemid' => $userid,
        'mimetype' => 'application/pdf',
        'contextid' => $contextid,
        'filename' => $name,
        'filepath' => '/',
        'userid' => $userid
    );
    
    $fs = get_file_storage();
    $fileResult = $DB->get_record('files', $fileinfo);
    if(!$fileResult) return false;
    
    $file_hash = $fileResult->pathnamehash;
    $file = $fs->get_file_by_hash($file_hash);

    return $file;
}

function get_signature_iframe($type, $cm_id, $pdf, $api_key='', $extra_params = []) {
    global $CFG_SIG, $CFG;
    $html = '';

    $embed = strpos($type, 'embed') !== false;
    $view = strpos($type, 'view') !== false;
    $sign = strpos($type, 'sign') !== false;

    $target = '';
    if($embed) {
        $target = 'signactivity_iframe_'.$cm_id;
    }

    if($view) {
        $src = $CFG_SIG->base_url.'v1/viewPdf';
    }
    else if($sign) {
        $src = $CFG_SIG->base_url.'v1/userSignature';
    }
    else {
        $src = $CFG_SIG->base_url.'v1/coordinatesPdf';
    }
    
    if($sign){
        $redirect_url = $CFG->wwwroot.'/mod/signactivityserver/save_signed_pdf.php';
    }
    else {
        $redirect_url = $CFG->wwwroot.'/mod/signactivityserver/save_encrypted_pdf.php';
    }
    

    $html .= '<form id="hidden_form_'.$cm_id.'" target="'.$target.'" action="'.$src.'" method="POST" style="display:none">';
    $html .= '<input type="text" name="id" value="'.$cm_id.'" />';
    $html .= '<input type="text" name="embed" value="1" />';
    $html .= '<input type="text" name="redirect_url" value="'.$redirect_url.'" />';
    $html .= '<input type="text" name="report_engine_value" value="'.$pdf.'" />';
    $html .= '<input type="text" name="api_key" value="'.$api_key.'" />';
    $html .= '<input type="text" name="referrer" value="'.$CFG->wwwroot.'" />';
    foreach($extra_params as $key => $value) {
        $html .= '<input type="text" name="'.$key.'" value="'.$value.'" />';
    }
    $html .= '</form>';
    
    if($embed) {
        $html .= '<iframe id="'.$target.'" name="'.$target.'" width="1000" height="1280" src="" style="visibility:hidden"></iframe>';
    }

    $html .= '
    <script>
        let target = "'.$target.'";
        window.addEventListener("load", function () {
            document.getElementById("hidden_form_'.$cm_id.'").submit(); 
            if(target) document.getElementById(target).style.visibility = "visible"; 
        })
    </script>';
       
    return $html;
}

function save_key($key) {
    global $USER, $DB;

    $old_key = $DB->get_record('signactivityserver_key', array('user_id' => $USER->id));

    if ($old_key) {
        $old_key->user_key = $key;
        $DB->update_record('signactivityserver_key', $old_key);
    } else {
        $rec = new stdClass();
        $rec->user_id = $USER->id;
        $rec->user_key = $key;
        $DB->insert_record('signactivityserver_key', $rec);
    }
}